/*------------------------------------------------------------------------------*
 * File Name: Report.h		 													*
 * Creation: ML and GJL 12/22/03												*
 * Purpose: OriginC header file containing report generation macros for use		*
 * 			with Worksheet::SetReportTree class member function.				*
 * Copyright (c) OriginLab Corp.	2000, 2001, 2002, 2003, 2004, 2005, 2006	*
 * All Rights Reserved															*
 * 																				*
 * Modification Log:															*
 *------------------------------------------------------------------------------*/

#define AttrStyle "Style"

/**
		Declare a report tree.
	Example:
		DECLARE_REPORT(Test)
	Creates:
		trReport_Test
*/
#define DECLARE_REPORT(_RPT_NAME) \
	Tree trReport_##_RPT_NAME;\
	TreeNode _tmpTree = trReport_##_RPT_NAME;\
	TreeNode _tmpNode, _tmpSubnode;\
	int _nCount, _nTable = 1;\
	string _strTemp;

/**
		Set the report title.
	Example:
		SET_REPORT_TITLE("Title of Report", "Heading1")
	Creates:
		trReport_Test.Title.strVal == "Title of Report"
		trReport_Test.Title.Style == "Heading1"
*/
#define SET_REPORT_TITLE(_TITLE, _STYLE) \
	_tmpTree.Title.strVal = _TITLE;\
	_tmpTree.Title.SetAttribute(AttrStyle, _STYLE);

/**
		Set the report subtitle.
	Example:
		SET_REPORT_SUBTITLE("Subtitle of Report", "Heading2")
	Creates:
		trReport_Test.Subtitle.strVal == "Subtitle of Report"
		trReport_Test.Subtitle.Style == "Heading2"
*/
#define SET_REPORT_SUBTITLE(_SUBTITLE, _STYLE) \
	_tmpTree.Subtitle.strVal = _SUBTITLE;\
	_tmpTree.Subtitle.SetAttribute(AttrStyle, _STYLE);

/**
		Add a table to a report.
	Example:
		MatrixLayer matLayer=Project.ActiveLayer();
		if(matLayer.IsValid())
		{
			Matrix mat(matLayer);
			if(mat.IsValid())
			 	ADD_TABLE_TO_REPORT(mat.GetName());
		}
	Creates:
		trReport_Test.Table1.Style == mat.GetName()
*/
#define ADD_TABLE_TO_REPORT(_STYLE) \
	_strTemp.Format("Table%d", _nTable++);\
	_tmpNode = _tmpTree.AddNode(_strTemp);\
	_tmpNode.SetAttribute(AttrStyle, _STYLE);\
	_nCount = 1;  

/**
		Set a table title.
	Example:
		SET_TABLE_TITLE("Title of Table1", "Heading3")
	Creates:
		trReport_Test.Table1.Title.strVal == "Title of Table1"
		trReport_Test.Table1.Title.Style == "Heading3"
*/
#define SET_TABLE_TITLE(_TITLE, _STYLE) \
	_tmpNode.Title.strVal = _TITLE;\
	_tmpNode.Title.SetAttribute(AttrStyle, _STYLE);

/**
		Set a table subtitle.
	Example:
		SET_TABLE_SUBTITLE("Subtitle of Table1", "Heading4")
	Creates:
		trReport_Test.Table1.Subtitle.strVal == "Subtitle of Table1"
		trReport_Test.Table1.Subtitle.Style == "Heading4"
*/
#define SET_TABLE_SUBTITLE(_SUBTITLE, _STYLE) \
	_tmpNode.Subtitle.strVal = _SUBTITLE;\
	_tmpNode.Subtitle.SetAttribute(AttrStyle, _STYLE);

/**
		Set column headings for a table.
	Example:
		vector<string> vColHead = {"Col1 Label","Col2 Label","Col3 Label","Col4 Label"};
		SET_TABLE_COLUMN_HEADINGS(vColHead)
	Creates:
		trReport_Test.Table1.ColHeading.strVals == {"Col1 Label","Col2 Label","Col3 Label","Col4 Label"}
*/
#define SET_TABLE_COLUMN_HEADINGS(_COLHEADINGS) \
	_tmpNode.ColHeading.strVals = _COLHEADINGS;

/**
		Set row headings for a table.
	Example:
		vector<string> vRowHead = {"Row1 Label","Row2 Label","Row3 Label","Row4 Label","Row5 Label"};
		SET_TABLE_ROW_HEADINGS(vRowHead)
	Creates:
		trReport_Test.Table1.RowHeading.strVals == {"Row1 Label","Row2 Label","Row3 Label","Row4 Label","Row5 Label"}
*/
#define SET_TABLE_ROW_HEADINGS(_ROWHEADINGS) \
	_tmpNode.RowHeading.strVals = _ROWHEADINGS;

/**
		Set matrix values for a table of doubles.
	Example:
		matrix mResults = {{1.1,1.2,1.3,1.4},{2.1,2.2,2.3,2.4},{3.1,3.2,3.3,3.4},{4.1,4.2,4.3,4.4},{5.1,5.2,5.3,5.4}};
		SET_TABLE_MATRIX_DVALS2(mResults)
	Creates:
		trReport_Test.Table1.Matrix.dVals2 == {{1.1,1.2,1.3,1.4},{2.1,2.2,2.3,2.4},{3.1,3.2,3.3,3.4},{4.1,4.2,4.3,4.4},{5.1,5.2,5.3,5.4}}
*/
#define SET_TABLE_MATRIX_DVALS2(_MATRIX) \
	_tmpNode.Matrix.dVals2 = _MATRIX;

/**
		Set matrix values for a table of ints.
	Example:
		matrix<int> mResults = {{11,12,13,14},{21,22,23,24},{31,32,33,34},{41,42,43,44},{51,52,53,54}};
		SET_TABLE_MATRIX_NVALS2(mResults)
	Creates:
		trReport_Test.Table1.Matrix.nVals2 == {{11,12,13,14},{21,22,23,24},{31,32,33,34},{41,42,43,44},{51,52,53,54}}
*/
#define SET_TABLE_MATRIX_NVALS2(_MATRIX) \
	_tmpNode.Matrix.nVals2 = _MATRIX;

/**
		Add a column of doubles to a table.
	Example:
		vector vResults = {1.0,2.0,3.0,4.0,5.0};
		ADD_DVALS_COLUMN_TO_TABLE(vResults, "Doubles1")
	Creates:
		trReport_Test.Table2.Col1.dVals == {1.0,2.0,3.0,4.0,5.0}
		trReport_Test.Table2.Col1.Style == "Doubles1"
*/	
#define ADD_DVALS_COLUMN_TO_TABLE(_DVALS, _STYLE) \
	_strTemp.Format("Col%d", _nCount++);\
	_tmpSubnode = _tmpNode.AddNode(_strTemp);\
	_tmpSubnode.dVals = _DVALS;\
	_tmpSubnode.SetAttribute(AttrStyle, _STYLE);

/**
		Add a column of strings to a table.
	Example:
		vector<string> vStrings = {"String1","String2","String3","String4","String5"};
		ADD_STRVALS_COLUMN_TO_TABLE(vStrings, "Strings1")
	Creates:
		trReport_Test.Table2.Col2.strVals == {"String1","String2","String3","String4","String5"}
		trReport_Test.Table2.Col2.Style == "Strings1"
*/	
#define ADD_STRVALS_COLUMN_TO_TABLE(_STRVALS, _STYLE) \
	_strTemp.Format("Col%d", _nCount++);\
	_tmpSubnode = _tmpNode.AddNode(_strTemp);\
	_tmpSubnode.strVals = _STRVALS;\
	_tmpSubnode.SetAttribute(AttrStyle, _STYLE);

/**
		Add a column of ints to a table.
	Example:
		vector<int> vYesNo = {1,0,1,1,0};
		ADD_NVALS_COLUMN_TO_TABLE(vYesNo, "Int1")
	Creates:
		trReport_Test.Table2.Col3.nVals == {1,0,1,1,0}
		trReport_Test.Table2.Col3.Style == "Int1"
*/	
#define ADD_NVALS_COLUMN_TO_TABLE(_NVALS, _STYLE) \
	_strTemp.Format("Col%d", _nCount++);\
	_tmpSubnode = _tmpNode.AddNode(_strTemp);\
	_tmpSubnode.nVals = _NVALS;\
	_tmpSubnode.SetAttribute(AttrStyle, _STYLE);

/**
		Add a categorical column to a table.
	Example:
		vector<int> vColors = {3,1,2,2,3};
		vector<string> vMap = {"Red","Green","Blue"};
		ADD_CATEGORICAL_COLUMN_TO_TABLE(vYesNo, vMap, "Categorical1")
	Creates:
		trReport_Test.Table3.Col4.CatVals.nVals == {3,1,2,2,3} // == Blue, Red, Green, Green, Blue
		trReport_Test.Table3.Col4.MapVals.Style == {"Red","Green","Blue"}
		trReport_Test.Table3.Col4.MapVals.Style == "Categorical1"
*/	
#define ADD_CATEGORICAL_COLUMN_TO_TABLE(_NVALS, _MAPVALS, _STYLE) \
	_strTemp.Format("Col%d", _nCount++);\
	_tmpSubnode = _tmpNode.AddNode(_strTemp);\
	_tmpSubnode.CatVals.nVals = _NVALS;\
	_tmpSubnode.MapVals.strVals = _MAPVALS;\
	_tmpSubnode.MapVals.SetAttribute(AttrStyle, _STYLE);

/**
		Add a row of doubles to a table.
	Example:
		vector vResults = {1.0,2.0,3.0,4.0,5.0};
		ADD_DVALS_ROW_TO_TABLE(vResults, "Doubles1")
	Creates:
		trReport_Test.Table3.Row1.dVals == {1.0,2.0,3.0,4.0,5.0}
		trReport_Test.Table3.Row1.Style == "Doubles1"
*/	
#define ADD_DVALS_ROW_TO_TABLE(_DVALS, _STYLE) \
	_strTemp.Format("Row%d", _nCount++);\
	_tmpSubnode = _tmpNode.AddNode(_strTemp);\
	_tmpSubnode.dVals = _DVALS;\
	_tmpSubnode.SetAttribute(AttrStyle, _STYLE);

/**
		Add a row of strings to a table.
	Example:
		vector<string> vStrings = {"String1","String2","String3","String4","String5"};
		ADD_STRVALS_ROW_TO_TABLE(vStrings, "Strings1")
	Creates:
		trReport_Test.Table3.Row2.strVals == {"String1","String2","String3","String4","String5"}
		trReport_Test.Table3.Row2.Style == "Strings1"
*/	
#define ADD_STRVALS_ROW_TO_TABLE(_STRVALS, _STYLE) \
	_strTemp.Format("Row%d", _nCount++);\
	_tmpSubnode = _tmpNode.AddNode(_strTemp);\
	_tmpSubnode.strVals = _STRVALS;\
	_tmpSubnode.SetAttribute(AttrStyle, _STYLE);

/**
		Add a row of ints to a table.
	Example:
		vector<int> vYesNo = {1,0,1,1,0};
		ADD_NVALS_ROW_TO_TABLE(vYesNo, "Int1")
	Creates:
		trReport_Test.Table3.Row3.nVals == {1,0,1,1,0}
		trReport_Test.Table3.Row3.Style == "Int1"
*/	
#define ADD_NVALS_ROW_TO_TABLE(_NVALS, _STYLE) \
	_strTemp.Format("Row%d", _nCount++);\
	_tmpSubnode = _tmpNode.AddNode(_strTemp);\
	_tmpSubnode.nVals = _NVALS;\
	_tmpSubnode.SetAttribute(AttrStyle, _STYLE);

/**
		Add a categorical row to a table.
	Example:
		vector<int> vColors = {3,1,2,2,3};
		vector<string> vMap = {"Red","Green","Blue"};
		ADD_CATEGORICAL_ROW_TO_TABLE(vYesNo, vMap, "Categorical1")
	Creates:
		trReport_Test.Table3.Row4.CatVals.nVals == {3,1,2,2,3} // == Blue, Red, Green, Green, Blue
		trReport_Test.Table3.Row4.MapVals.Style == {"Red","Green","Blue"}
		trReport_Test.Table3.Row4.MapVals.Style == "Categorical1"
*/	
#define ADD_CATEGORICAL_ROW_TO_TABLE(_NVALS, _MAPVALS, _STYLE) \
	_strTemp.Format("Row%d", _nCount++);\
	_tmpSubnode = _tmpNode.AddNode(_strTemp);\
	_tmpSubnode.CatVals.nVals = _NVALS;\
	_tmpSubnode.MapVals.strVals = _MAPVALS;\
	_tmpSubnode.MapVals.SetAttribute(AttrStyle, _STYLE);
